﻿using System;
using System.Collections.Generic;
using System.Data.Linq;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Threading.Tasks;
using VeteransAffairs.Registries.BusinessAHOBPR;
using VeteransAffairs.Registries.BusinessAHOBPR.PaymentService;
using VeteransAffairs.Registries.BusinessAHOBPR.Extensions;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR.Emis
{
    public class RegistrantDeploymentManager : AHOBPRBaseBO, IUpdateManager
    {
        string connectionInfo = string.Empty;
        RegistrantDeploymentUtilities regDeploymentUtilities;
        public RegistrantDeploymentManager() { }
        public RegistrantDeploymentManager(string dbConnectionInfo)
        {
            connectionInfo = dbConnectionInfo;
        }

        public string Name { get { return "DeploymentImporter"; } }
        public RegistrantDeploymentUtilities RegDeploymentUtilities
        {
            get {
                if (regDeploymentUtilities == null)
                {
                    regDeploymentUtilities = new RegistrantDeploymentUtilities(connectionInfo);
                }
                return regDeploymentUtilities;
            }
        }
        private void AddDeployments(AHOBPRDataAccess db, List<REGISTRANT_DEPLOYMENT> deploymentsToAdd)
        {
            db.REGISTRANT_DEPLOYMENTs.InsertAllOnSubmit(deploymentsToAdd);
            db.SubmitChanges();
        }
        static IEnumerable<REGISTRANT_DEPLOYMENT> CheckForDupes(List<REGISTRANT_DEPLOYMENT> existingDeployments, REGISTRANT_DEPLOYMENT b)
        {
            //Some countries SOMETIMES have a period (.) at the end of their name. Other times they don't. Using TrimEnd to deal with this.
            char[] MyChar = { '.', ' ' };
            var x = existingDeployments.AsEnumerable().Where(a =>
                                                a.DEPLOYMENT_START_DATE.Value.Date == b.DEPLOYMENT_START_DATE.Value.Date
                                                && a.DEPLOYMENT_END_DATE.Value.Date == b.DEPLOYMENT_END_DATE.Value.Date
                                                && a.DEPLOYMENT_COUNTRY.TrimEnd(MyChar) == b.DEPLOYMENT_COUNTRY.TrimEnd(MyChar)
                                                && a.STD_BRANCH_OF_SERVICE_ID == b.STD_BRANCH_OF_SERVICE_ID);
            //Console.WriteLine($"# of Registrant Deployments: {existingDeployments.Count()}. -- Dupes: {x.Count() }");
            return x;
        }

        private List<REGISTRANT_DEPLOYMENT> GetDeploymentsFromDb(AHOBPRDataAccess db, int registrantId)
        {
            //db.DeferredLoadingEnabled = false;
            var result = db.REGISTRANT_DEPLOYMENTs.Where(d => d.REGISTRANT_ID == registrantId).ToList();
            return result;
        }

        private void DeleteRegistrantServiceOccupations(int deploymentId, AHOBPRDataAccess db)
        {
            var entries = db.REGISTRANT_SERVICE_SPECIFIC_OCCUPATIONs.Where(e => e.REGISTRANT_DEPLOYMENT_ID == deploymentId);
            db.REGISTRANT_SERVICE_SPECIFIC_OCCUPATIONs.DeleteAllOnSubmit(entries);
            db.SubmitChanges();
        }
        private void AddServiceOccupationsToDeployment(REGISTRANT_DEPLOYMENT deployment, RegistrantHistory history, AHOBPRDataAccess db)
        {
            var deploymentDateRange = new DateRange(deployment.DEPLOYMENT_START_DATE.Value.Date, deployment.DEPLOYMENT_END_DATE.Value.Date);
            var occupationDate = history.OccupationDateTimesByDateRange.FirstOrDefault(e => e.Key.HasOverlap(deploymentDateRange)).Value;

            if (occupationDate != DateTime.MinValue)
            {
                var serviceSpecificOccupations = history.OccupationsByDateTime[occupationDate];
                foreach (var sso in serviceSpecificOccupations)
                {
                    try
                    {
                        REGISTRANT_SERVICE_SPECIFIC_OCCUPATION rsso = sso.ToRSSO();
                        rsso.REGISTRANT_DEPLOYMENT = deployment;
                        db.REGISTRANT_SERVICE_SPECIFIC_OCCUPATIONs.InsertOnSubmit(rsso);
                    }
                    catch (Exception ex)
                    {
                        AHOBPRLogger.LogErrorMessage("Exception", "RegistrantDeploymentManager.AddServiceOccupationsToDeployment()", ex.Message);
                    }
                }
            }
        }
        public bool ProcessDeployments(List<IDeployment> deploymentsFromEmis, int registrantId, RegistrantHistory history)
        {
            bool result = true;
            var additions = new List<REGISTRANT_DEPLOYMENT>();
            var updates = new List<REGISTRANT_DEPLOYMENT>();
            try
            {
                using (var db = GetLocalContext(connectionInfo))
                {
                    var registrant = db.REGISTRANTs.FirstOrDefault(r => r.REGISTRANT_ID == registrantId);

                    if (registrant != null)
                    {
                        var existingDeploymentsFromDb = GetDeploymentsFromDb(db, registrantId);
                        List<int> possibleRegistrantEnteredDeployments = ProcessEmisDeployments(deploymentsFromEmis, registrant, additions, db, existingDeploymentsFromDb);
                        if (possibleRegistrantEnteredDeployments.Any())
                        {
                            RegistrantHistoryToDeploymentMapper mapper = new RegistrantHistoryToDeploymentMapper();

                            foreach (var deploymentId in possibleRegistrantEnteredDeployments)
                            {
                                var deploymentToUpdate = existingDeploymentsFromDb.First(e => e.REGISTRANT_DEPLOYMENT_ID == deploymentId);
                                try
                                {
                                    DeleteRegistrantServiceOccupations(deploymentId, db);
                                    AddServiceOccupationsToDeployment(deploymentToUpdate, history, db);
                                    SetPayPlan(history, deploymentToUpdate);
                                    SetStdComponentId(history, deploymentToUpdate);
                                    RegDeploymentUtilities.SetDeploymentStage(deploymentToUpdate, registrant.STD_REGISTRANT_STATUS_ID);
                                    deploymentToUpdate.UPDATEDBY = Name;
                                }
                                catch (NullReferenceException ex)
                                {
                                    Trace.WriteLine($"RegistrantDeploymentManager.ProcessDeployments().Foreach: {ex.Message}");
                                    AHOBPRLogger.LogErrorMessage("Exception", "RegistrantDeploymentManager.ProcessDeployments().foreach", $"RegistrantDeploymentManager.ProcessDeployments().foreach: {ex.Message}");
                                    result = false;
                                }
                            }
                            db.SubmitChanges();
                        }
                    }
                    else
                    {
                        AHOBPRLogger.LogErrorMessage("INFO", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, $"Made a call for registrant with registrant_id of {registrantId}, but no matching registrant with that ID was found in database!");
                    }
                }
            }
            catch (Exception ex)
            {
                Trace.WriteLine($"RegistrantDeploymentManager.ProcessDeployments(): {ex.Message}");
                AHOBPRLogger.LogErrorMessage("Exception", "RegistrantDeploymentManager.ProcessDeployments()", $"RegistrantDeploymentManager.ProcessDeployments(): {ex.Message}");
                result = false;
            }         

            return result;
        }
        private void SetStdComponentId(RegistrantHistory history, REGISTRANT_DEPLOYMENT deployment)
        {
            var dateRange = new DateRange(deployment.DEPLOYMENT_START_DATE.Value, deployment.DEPLOYMENT_END_DATE.Value);
            var code = history.GetPersonnelCategoryCode(dateRange);
            deployment.STD_COMPONENT_ID = !string.IsNullOrEmpty(code) ? AHOBPRShared.GetStdComponent(code) : -1;
        }

        private List<int> ProcessEmisDeployments(List<IDeployment> deploymentsFromEmis, REGISTRANT registrant, List<REGISTRANT_DEPLOYMENT> additions, AHOBPRDataAccess db, List<REGISTRANT_DEPLOYMENT> existingDeployments)
        {
            var possibleRegistrantEnteredDeployments = existingDeployments.Select(e => e.REGISTRANT_DEPLOYMENT_ID).ToList();


            foreach (REGISTRANT_DEPLOYMENT emisDeployment in deploymentsFromEmis)
            {
                //Check if it exists in registrant's deployment list
                IEnumerable<REGISTRANT_DEPLOYMENT> matchingDeploymentsInDB = CheckForDupes(existingDeployments, emisDeployment);
                var numberOfMatchingDeployments = (matchingDeploymentsInDB != null) ? matchingDeploymentsInDB.Count() : 0;
                if (numberOfMatchingDeployments > 0)
                {
                    try
                    {
                        //Update Deployments
                        foreach (var deploymentInDb in matchingDeploymentsInDB)
                        {
                            //If it's in the DB, then it's not a new deployment entered by a registrant.
                            possibleRegistrantEnteredDeployments.Remove(deploymentInDb.REGISTRANT_DEPLOYMENT_ID);
                            DeleteRegistrantServiceOccupations(deploymentInDb.REGISTRANT_DEPLOYMENT_ID, db);
                            deploymentInDb.REGISTRANT_SERVICE_SPECIFIC_OCCUPATIONs.AddRange(emisDeployment.REGISTRANT_SERVICE_SPECIFIC_OCCUPATIONs);
                            //[STD_COMPONENT_ID]
                            deploymentInDb.STD_COMPONENT_ID = emisDeployment.STD_COMPONENT_ID;
                            //,[STD_PAYPLAN_ID]
                            deploymentInDb.STD_PAYPLAN_ID = emisDeployment.STD_PAYPLAN_ID;
                            RegDeploymentUtilities.SetDeploymentStage(deploymentInDb, registrant.STD_REGISTRANT_STATUS_ID);
                            deploymentInDb.UPDATED = DateTime.Now;
                            deploymentInDb.UPDATEDBY = Name;
                            db.SubmitChanges();
                        }
                    }
                    catch (NullReferenceException ex)
                    {
                        Trace.WriteLine($"RegistrantDeploymentManager.ProcessEmisDeployments: {ex.Message}");
                        AHOBPRLogger.LogErrorMessage("NullReferenceException", "RegistrantDeploymentManager.ProcessEmisDeployments", $"RegistrantDeploymentManager.ProcessEmisDeployments: {ex.Message}");
                    }
                }
                else
                {
                    //Not in database, Add to Deployments List to be passed back
                    emisDeployment.REGISTRANT_ID = registrant.REGISTRANT_ID;
                    RegDeploymentUtilities.SetDeploymentStage(emisDeployment, registrant.STD_REGISTRANT_STATUS_ID);
                    emisDeployment.CREATEDBY = this.Name;
                    emisDeployment.UPDATEDBY = this.Name;
                    additions.Add(emisDeployment);
                }

            }

            if (additions.Any()) { AddDeployments(db, additions); }

            return possibleRegistrantEnteredDeployments;
        }
        private void SetPayPlan(RegistrantHistory history, REGISTRANT_DEPLOYMENT deployment)
        {
            var deploymentDateRange = new DateRange(deployment.DEPLOYMENT_START_DATE.Value, deployment.DEPLOYMENT_END_DATE.Value);
            var bosCode = history.GetBranchOfServiceCode(deploymentDateRange);
            var payData = history.GetPayHistoryData(deploymentDateRange);
            var payPlan = new PaymentHistoryManager(connectionInfo).LookupPayPlan(payData, bosCode);
            deployment.STD_PAYPLAN_ID = payPlan.STD_PAYPLAN_ID;
        }
        public bool Update(string edipi, int registrantId)
        {
            return UpdateRegistrantDeployments(edipi, registrantId);
        }
        private bool UpdateRegistrantDeployments(string edipi, int registrantId)
        {
            var result = true;
            try
            {
                var factory = new RegistrantDeploymentFactory();
                var historyManager = new RegistrantHistoryManager(connectionInfo);
                var history = historyManager.CreateRegistrantHistory(edipi);
                List<IDeployment> deploymentsFromEmis = new DoDDeploymentImporter().GetRegistrantDeployments(history, edipi, factory);
                result = ProcessDeployments(deploymentsFromEmis, registrantId, history);
            }
            catch (Exception ex)
            {
                AHOBPRLogger.LogErrorMessage("Exception", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message + "; " + ex.StackTrace);
                result = false;
            }

            return result;
        }
    }
}
